import React, { useEffect, useMemo, useState } from "react";
import Navbar from "../components/Navbar";
import ProgressBar from "../components/ProgressBar";
import PuzzleBoard from "../components/PuzzleBoard";
import { usePuzzleStore } from "../store/puzzleStore";

export default function Home() {
  const {
    levels,
    levelIndex,
    initPuzzles,
    initialized,
    nextQuestion,
    resetAll,
    saveRun,
  } = usePuzzleStore();

  const [qIndex, setQIndex] = useState(0);
  const [playing, setPlaying] = useState(false);
  const [finishedRun, setFinishedRun] = useState(false);

  // 🧩 Initialize puzzles on first load
  useEffect(() => {
    if (!initialized) initPuzzles();
  }, []);

  const currentLevel = levels[levelIndex] || null;
  const questions = currentLevel?.questions || [];

  // Reset question index whenever level changes
  useEffect(() => {
    setQIndex(0);
  }, [levelIndex]);

  // Current question memoized
  const currentQuestion = useMemo(() => {
    if (questions[qIndex]) {
      return {
        ...questions[qIndex],
        timeLimit: currentLevel?.timeLimit || 10,
      };
    }
    return null;
  }, [questions, qIndex, currentLevel]);

  // ✅ Handle correct/wrong/times-up → next step
  function handleNext(correct, timeTaken) {
    const totalLevels = levels.length;

    // Move to next question
    if (qIndex + 1 < questions.length) {
      setQIndex((prev) => prev + 1);
    } else {
      // Move to next level
      if (levelIndex + 1 < totalLevels) {
        nextQuestion();
        setQIndex(0);
      } else {
        // ✅ Finished ALL levels
        setFinishedRun(true);
        setPlaying(false);
      }
    }
  }

  // 🏁 Save run result to leaderboard
  function handleFinishAndSave() {
    const name =
      prompt("Enter your name for leaderboard:", "Player") || "Player";
    saveRun(name);
    resetAll();
    setPlaying(false);
    setFinishedRun(false);
    alert("✅ Run saved to local leaderboard!");
  }

  // 🔄 Reset everything
  function handleReset() {
    resetAll();
    setPlaying(false);
    setFinishedRun(false);
    setQIndex(0);
  }

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-1 container mx-auto px-6 py-8 flex flex-col items-center">
        <div className="w-full max-w-4xl space-y-6">
          {/* ===== Progress Bar ===== */}
          {!finishedRun && (
            <div className="glass p-4 rounded-lg">
              <ProgressBar
                value={levelIndex}
                max={Math.max(1, levels.length)}
              />
              <div className="mt-2 text-sm text-white/70">
                Level {Math.min(levels.length, levelIndex + 1)} /{" "}
                {levels.length}
              </div>
            </div>
          )}

          {finishedRun && (
            <div className="glass p-4 rounded-lg text-center">
              <div className="w-full h-2 bg-gradient-to-r from-indigo-400 to-purple-500 rounded-full mb-2"></div>
              <div className="text-sm text-white/80 font-medium">
                ✅ All Levels Completed!
              </div>
            </div>
          )}

          <div className="flex flex-col md:flex-row gap-6">
            <div className="flex-1">
              {/* ===== Level Info ===== */}
              <div className="glass p-6 rounded-2xl">
                <h2 className="text-xl font-semibold">
                  {currentLevel ? currentLevel.name : "Loading..."}
                </h2>
                <p className="text-sm text-white/70 mt-2">
                  {currentLevel
                    ? `${currentLevel.questions?.length} questions • ${currentLevel.timeLimit}s each`
                    : ""}
                </p>

                <div className="mt-4 flex gap-3">
                  {!playing && !finishedRun && (
                    <button
                      onClick={() => setPlaying(true)}
                      className="px-4 py-2 bg-indigo-600 rounded hover:bg-indigo-700 transition-all"
                    >
                      Start
                    </button>
                  )}
                  <button
                    onClick={handleReset}
                    className="px-4 py-2 bg-white/10 rounded hover:bg-white/20 transition-all"
                  >
                    Reset
                  </button>
                </div>
              </div>

              {/* ===== Puzzle Area ===== */}
              <div className="mt-6">
                {/* Playing questions */}
                {playing && currentQuestion && !finishedRun && (
                  <PuzzleBoard question={currentQuestion} onNext={handleNext} />
                )}

                {/* ===== Finished Run ===== */}
                {finishedRun && (
                  <div className="glass p-6 rounded-2xl mt-6 text-center animate-fadeIn">
                    <h3 className="text-2xl font-bold">🏁 Run Finished!</h3>
                    <p className="text-white/70 mt-2">
                      Great job completing all levels! You can save your score
                      or start again.
                    </p>
                    <div className="mt-4 flex justify-center gap-3">
                      <button
                        onClick={handleFinishAndSave}
                        className="px-4 py-2 bg-emerald-500 rounded hover:bg-emerald-600 transition-all"
                      >
                        Save Run
                      </button>
                      <button
                        onClick={handleReset}
                        className="px-4 py-2 bg-white/10 rounded hover:bg-white/20 transition-all"
                      >
                        Restart
                      </button>
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* ===== Sidebar Stats ===== */}
            <aside className="w-full md:w-72">
              <div className="glass p-4 rounded-lg space-y-4">
                <StatBlock
                  title="Streak"
                  value={usePuzzleStore.getState().streak}
                />
                <StatBlock
                  title="Correct"
                  value={usePuzzleStore.getState().correctCount}
                />
                <StatBlock
                  title="Wrong"
                  value={usePuzzleStore.getState().wrongCount}
                />
                <StatBlock
                  title="Total Time"
                  value={`${usePuzzleStore.getState().totalTime}s`}
                />
              </div>
            </aside>
          </div>
        </div>
      </main>
    </div>
  );
}

// ===== Stat Block Component =====
function StatBlock({ title, value }) {
  return (
    <div>
      <div className="text-sm text-white/70">{title}</div>
      <div className="text-2xl font-semibold">{value}</div>
    </div>
  );
}
